from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.evidences import ServiceEvidence
from codeable_detectors.pyparsing_patterns import ID, round_braces_block
from codeable_detectors.python.pythonDetectors import detect_python_import, FailedEvidence, \
    get_import_patterns_from_imports_evidences


class PythonFlaskRestfulService(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["py"]

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        is_detected = False

        import_matches = detect_python_import(ctx, "flask", "Flask")
        if not import_matches:
            return FailedEvidence("python import for flask failed")
        matches.extend(import_matches)

        for import_pattern in get_import_patterns_from_imports_evidences(import_matches):
            # match something like: app = Flask(__name__)
            flask_app_matches = ctx.matches_pattern(ID + Literal("=") + import_pattern + round_braces_block)
            if not flask_app_matches:
                return FailedEvidence("cannot find instantiation of Flask app")

            for flask_app_match in flask_app_matches:
                matches.append(flask_app_match)
                flask_var_name = flask_app_match.text[:flask_app_match.text.find("=")].strip()
                print("###flask_var_name= " + flask_var_name)
                flask_route_def_matches = ctx.match_indented_python_block(Literal("@" + flask_var_name) + Literal(".") +
                                                                          Literal("route") + round_braces_block +
                                                                          Literal("def") + ID + round_braces_block +
                                                                          Literal(":"))
                # we could now check for jsonify and other possible returns in the indented block
                # but this is hard to impossible to automate, e.g. just returning a dict also creates
                # JSON and this can be created in many possible ways
                # => a human needs to inspect there is actually a restful service, and not e.g. 
                # a web http app.route
                if len(flask_route_def_matches) > 0:
                    matches.extend(flask_route_def_matches)
                    is_detected = True

        if is_detected:
            return ServiceEvidence(matches).set_properties(
                detector_technology_types=["python", "flask", "restful"], kwargs=kwargs)
        return FailedEvidence("cannot find method with a Flask route annotation")
